#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <bpf/libbpf.h>
#include "stmmac_bpf_trace.skel.h"
#include "stmmac_bpf_trace.h"

static struct stmmac_bpf_trace_bpf *skel;
static struct bpf_link *fentry_link = NULL;

static int libbpf_print_fn(enum libbpf_print_level level, const char *format, va_list args)
{
    return vfprintf(stderr, format, args);
}

static void sig_handler(int sig)
{
    fprintf(stderr, "Exiting...\n");
    if (fentry_link) {
        bpf_link__destroy(fentry_link);
        fentry_link = NULL;
    }
    if (skel) {
        stmmac_bpf_trace_bpf__destroy(skel);
        skel = NULL;
    }
    exit(0);
}

static int handle_event(void *ctx, void *data, size_t data_sz)
{
    const struct event *e = data;
    const char *command_str;
    
    if (e->is_xdp_setup_prog) {
        command_str = "XDP_SETUP_PROG";
    } else if (e->is_xdp_setup_xsk_pool) {
        command_str = "XDP_SETUP_XSK_POOL";
    } else {
        command_str = "UNKNOWN";
    }
    
    printf("stmmac_bpf called:\n");
    printf("  dev: 0x%lx\n", e->dev_ptr);
    printf("  bpf: 0x%lx\n", e->bpf_ptr);
    printf("  command: %u (%s)\n", e->command, command_str);
    
    if (e->is_unknown) {
        printf("  WARNING: Unknown command received!\n");
    }
    printf("\n");
    
    return 0;
}

int main(int argc, char **argv)
{
    struct bpf_program *prog;
    struct ring_buffer *rb = NULL;
    int err;
    
    libbpf_set_print(libbpf_print_fn);
    
    signal(SIGINT, sig_handler);
    signal(SIGTERM, sig_handler);
    
    skel = stmmac_bpf_trace_bpf__open();
    if (!skel) {
        fprintf(stderr, "Failed to open BPF skeleton\n");
        return 1;
    }
    
    err = stmmac_bpf_trace_bpf__load(skel);
    if (err) {
        fprintf(stderr, "Failed to load BPF skeleton: %d\n", err);
        goto cleanup;
    }
    
    prog = bpf_object__find_program_by_name(skel->obj, "stmmac_bpf_entry");
    if (!prog) {
        fprintf(stderr, "Failed to find BPF program\n");
        err = -ENOENT;
        goto cleanup;
    }
    
    fentry_link = bpf_program__attach(prog);
    if (!fentry_link) {
        fprintf(stderr, "Failed to attach BPF program: %d\n", errno);
        err = -errno;
        goto cleanup;
    }
    
    rb = ring_buffer__new(bpf_map__fd(skel->maps.events), handle_event, NULL, NULL);
    if (!rb) {
        fprintf(stderr, "Failed to create ring buffer\n");
        err = -errno;
        goto cleanup;
    }
    
    printf("Tracing stmmac_bpf() commands... Ctrl-C to exit\n");
    printf("Looking for XDP_SETUP_PROG (%d) and XDP_SETUP_XSK_POOL (%d)\n\n", 
           XDP_SETUP_PROG, XDP_SETUP_XSK_POOL);
    
    while (true) {
        err = ring_buffer__poll(rb, 100);
        if (err == -EINTR) {  // Fixed: changed EINTERRUPT to EINTR
            err = 0;
            break;
        }
        if (err < 0) {
            fprintf(stderr, "Error polling ring buffer: %d\n", err);
            break;
        }
    }
    
    ring_buffer__free(rb);
    
cleanup:
    sig_handler(0);
    return err < 0 ? -err : 0;
}
