// stackcount.c - Count events and their stack traces using libbpf
// Based on BCC's stackcount.py

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <signal.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/resource.h>
#include <stdint.h>
#include <bpf/libbpf.h>
#include <bpf/bpf.h>
#include "stackcount.h"
#include "stackcount.skel.h"

static volatile bool exiting = false;

static void sig_handler(int sig)
{
    exiting = true;
}

static int libbpf_print_fn(enum libbpf_print_level level, const char *format, va_list args)
{
    return vfprintf(stderr, format, args);
}

static void print_frame(struct stackcount_bpf *obj, __u64 addr, pid_t pid, bool kernel, bool verbose, bool offset)
{
    char sym[256];
    
    printf("  ");
    if (verbose)
        printf("%-16llx ", addr);
    
    if (kernel) {
        if (offset) {
            if (!bpf_program__ksym_find(obj->progs.trace_count, addr, sym, sizeof(sym), NULL))
                printf("%s", sym);
        } else {
            if (!bpf_program__ksym_find(obj->progs.trace_count, addr, sym, sizeof(sym), NULL))
                printf("%s", sym);
        }
    } else {
        if (offset) {
            if (!bpf_program__sym_find(obj->progs.trace_count, addr, pid, sym, sizeof(sym), NULL))
                printf("%s", sym);
        } else {
            if (!bpf_program__sym_find(obj->progs.trace_count, addr, pid, sym, sizeof(sym), NULL))
                printf("%s", sym);
        }
    }
    printf("\n");
}

static void print_comm(const char *comm, pid_t pid)
{
    printf("    %s [%d]\n", comm, pid);
}

int main(int argc, char **argv)
{
    struct stackcount_bpf *obj;
    int err, i;
    bool kernel_stack = true;
    bool user_stack = true;
    bool verbose = false;
    bool offset = false;
    bool delimited = false;
    bool folded = false;
    bool timestamp = false;
    bool perpid = false;
    int pid = -1;
    int cpu = -1;
    int interval = 99999999;
    int duration = 0;
    char *pattern = NULL;
    
    // Parse command line arguments
    for (i = 1; i < argc; i++) {
        if (strcmp(argv[i], "-p") == 0) {
            if (++i >= argc) {
                fprintf(stderr, "Expected PID after -p\n");
                return 1;
            }
            pid = atoi(argv[i]);
            perpid = true;
        } else if (strcmp(argv[i], "-c") == 0) {
            if (++i >= argc) {
                fprintf(stderr, "Expected CPU after -c\n");
                return 1;
            }
            cpu = atoi(argv[i]);
        } else if (strcmp(argv[i], "-i") == 0) {
            if (++i >= argc) {
                fprintf(stderr, "Expected interval after -i\n");
                return 1;
            }
            interval = atoi(argv[i]);
        } else if (strcmp(argv[i], "-D") == 0) {
            if (++i >= argc) {
                fprintf(stderr, "Expected duration after -D\n");
                return 1;
            }
            duration = atoi(argv[i]);
        } else if (strcmp(argv[i], "-T") == 0) {
            timestamp = true;
        } else if (strcmp(argv[i], "-v") == 0) {
            verbose = true;
        } else if (strcmp(argv[i], "-d") == 0) {
            delimited = true;
        } else if (strcmp(argv[i], "-f") == 0) {
            folded = true;
        } else if (strcmp(argv[i], "-s") == 0) {
            offset = true;
        } else if (strcmp(argv[i], "-P") == 0) {
            perpid = true;
        } else if (strcmp(argv[i], "-K") == 0) {
            user_stack = false;
        } else if (strcmp(argv[i], "-U") == 0) {
            kernel_stack = false;
        } else if (argv[i][0] != '-') {
            pattern = argv[i];
        } else {
            fprintf(stderr, "Unknown option: %s\n", argv[i]);
            return 1;
        }
    }
    
    if (!pattern) {
        fprintf(stderr, "Must specify a pattern\n");
        return 1;
    }
    
    if (duration && !interval)
        interval = duration;
    
    libbpf_set_print(libbpf_print_fn);
    
    signal(SIGINT, sig_handler);
    signal(SIGTERM, sig_handler);
    
    // Load and verify BPF application
    obj = stackcount_bpf__open();
    if (!obj) {
        fprintf(stderr, "Failed to open BPF object\n");
        return 1;
    }
    
    // Set program parameters based on command line
    obj->rodata->filter_pid = pid;
    obj->rodata->filter_cpu = cpu;
    obj->rodata->perpid = perpid;
    obj->rodata->kernel_stack = kernel_stack;
    obj->rodata->user_stack = user_stack;
    
    // Load BPF program
    err = stackcount_bpf__load(obj);
    if (err) {
        fprintf(stderr, "Failed to load BPF object: %d\n", err);
        goto cleanup;
    }
    
    // Attach probes based on pattern
    if (strncmp(pattern, "t:", 2) == 0) {
        // Tracepoint
        char *tp = pattern + 2;
        char *colon = strchr(tp, ':');
        if (!colon) {
            fprintf(stderr, "Invalid tracepoint format, expected t:category:event\n");
            goto cleanup;
        }
        *colon = '\0';
        err = bpf_program__attach_tracepoint(obj->progs.trace_count, tp, colon+1);
        if (err) {
            fprintf(stderr, "Failed to attach tracepoint: %d\n", err);
            goto cleanup;
        }
    } else if (strncmp(pattern, "p:", 2) == 0 || strchr(pattern, ':')) {
        // Uprobe
        char *lib, *func;
        if (strncmp(pattern, "p:", 2) == 0) {
            lib = pattern + 2;
            func = strchr(lib, ':');
            if (!func) {
                fprintf(stderr, "Invalid uprobe format, expected p:lib:func\n");
                goto cleanup;
            }
            *func = '\0';
            func++;
        } else {
            lib = pattern;
            func = strchr(lib, ':');
            if (!func) {
                fprintf(stderr, "Invalid uprobe format, expected lib:func\n");
                goto cleanup;
            }
            *func = '\0';
            func++;
        }
        
        // TODO: Find library path
        err = bpf_program__attach_uprobe(obj->progs.trace_count, false, pid, lib, 0, func);
        if (err < 0) {
            fprintf(stderr, "Failed to attach uprobe: %d\n", err);
            goto cleanup;
        }
    } else {
        // Kprobe
        err = bpf_program__attach_kprobe(obj->progs.trace_count, false, pattern);
        if (err) {
            fprintf(stderr, "Failed to attach kprobe: %d\n", err);
            goto cleanup;
        }
    }
    
    if (!folded)
        printf("Tracing... Hit Ctrl-C to end.\n");
    
    while (!exiting) {
        sleep(interval);
        
        if (timestamp)
            printf("%-8s\n", "HH:MM:SS"); // TODO: Format actual time
        
        // TODO: Read and print stack counts
        // This would involve:
        // 1. Iterating through the counts hash map
        // 2. For each entry, walk the stack traces
        // 3. Print according to output format options
        
        if (duration && (interval * seconds) >= duration)
            break;
    }
    
    if (!folded)
        printf("Detaching...\n");

cleanup:
    stackcount_bpf__destroy(obj);
    return err != 0;
}
