#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <bpf/libbpf.h>
#include "page_pool_stats.skel.h"
#include "page_pool_stats.h"

static struct page_pool_stats_bpf *skel;
static struct bpf_link *fentry_link = NULL;

static int libbpf_print_fn(enum libbpf_print_level level, const char *format, va_list args)
{
    return vfprintf(stderr, format, args);
}

static void sig_handler(int sig)
{
    fprintf(stderr, "Exiting...\n");
    if (fentry_link) {
        bpf_link__destroy(fentry_link);
        fentry_link = NULL;
    }
    if (skel) {
        page_pool_stats_bpf__destroy(skel);
        skel = NULL;
    }
    exit(0);
}

static int handle_event(void *ctx, void *data, size_t data_sz)
{
    const struct event *e = data;
    
    printf("Page Pool: 0x%lx\n", e->pool_ptr);
    printf("  netdev pointer: 0x%lx\n", e->netdev_ptr);
    printf("  is NULL: %s\n", e->is_null ? "YES" : "NO");
    printf("  is NET_PTR_POISON: %s\n", e->is_poison ? "YES" : "NO");
    printf("  condition (!netdev || netdev == NET_PTR_POISON): %s\n", 
           e->condition_result ? "TRUE" : "FALSE");
    printf("\n");
    
    return 0;
}

int main(int argc, char **argv)
{
    struct bpf_program *prog;
    struct ring_buffer *rb = NULL;
    int err;
    
    libbpf_set_print(libbpf_print_fn);
    
    signal(SIGINT, sig_handler);
    signal(SIGTERM, sig_handler);
    
    skel = page_pool_stats_bpf__open();
    if (!skel) {
        fprintf(stderr, "Failed to open BPF skeleton\n");
        return 1;
    }
    
    err = page_pool_stats_bpf__load(skel);
    if (err) {
        fprintf(stderr, "Failed to load BPF skeleton: %d\n", err);
        goto cleanup;
    }
    
    prog = bpf_object__find_program_by_name(skel->obj, "page_pool_release_retry_entry");
    if (!prog) {
        fprintf(stderr, "Failed to find BPF program\n");
        err = -ENOENT;
        goto cleanup;
    }
    
    fentry_link = bpf_program__attach(prog);
    if (!fentry_link) {
        fprintf(stderr, "Failed to attach BPF program: %d\n", errno);
        err = -errno;
        goto cleanup;
    }
    
    rb = ring_buffer__new(bpf_map__fd(skel->maps.events), handle_event, NULL, NULL);
    if (!rb) {
        fprintf(stderr, "Failed to create ring buffer\n");
        err = -errno;
        goto cleanup;
    }
    
    printf("Tracing page_pool_release_retry() netdev condition... Ctrl-C to exit\n\n");
    
    while (true) {
        err = ring_buffer__poll(rb, 100);
        if (err == -EINTR) {
            err = 0;
            break;
        }
        if (err < 0) {
            fprintf(stderr, "Error polling ring buffer: %d\n", err);
            break;
        }
    }
    
    ring_buffer__free(rb);
    
cleanup:
    sig_handler(0);
    return err < 0 ? -err : 0;
}
